import logging
import numpy as np
import pandas as pd
from typing import Dict, List, Tuple, Any
import math
import json
from os.path import abspath, join
from scipy.stats import truncweibull_min

def load_json(path):
    return json.load(open(path, encoding='utf-8'))


def save_json(path, data):
    with open(path, 'w', encoding='utf-8') as out:
        json.dump(data, out, ensure_ascii=False, indent=4)


def load_solution(path):
    # Loads a solution from a json file to 2 pandas DataFrames.
    solution = load_json(path)
    fleet = pd.DataFrame(solution['fleet'])
    pricing_strategy = pd.DataFrame(solution['pricing_strategy'])
    return fleet, pricing_strategy


def save_solution(fleet, pricing_strategy, path):
    # Saves a solution into a json file.
    fleet = fleet.to_dict('records')
    pricing_strategy = pricing_strategy.to_dict('records')
    solution = {'fleet': fleet,
                'pricing_strategy': pricing_strategy}
    return save_json(path, solution)


def load_problem_data(path=None):
    if path is None:
        path = './data/'

    # LOAD DEMAND
    p = abspath(join(path, 'demand.csv'))
    demand = pd.read_csv(p)    
    
    # LOAD DATACENTERS DATA
    p = abspath(join(path, 'datacenters.csv'))
    datacenters = pd.read_csv(p)
    
    # LOAD SERVERS DATA
    p = abspath(join(path, 'servers.csv'))
    servers = pd.read_csv(p)
    
    # LOAD SELLING PRICES DATA
    p = abspath(join(path, 'selling_prices.csv'))
    selling_prices = pd.read_csv(p)

    # LOAD ELASTICITY DATA
    p = abspath(join(path, 'price_elasticity_of_demand.csv'))
    elasticity = pd.read_csv(p)
    return demand, datacenters, servers, selling_prices, elasticity




# CREATE LOGGER
logger = logging.getLogger()
# 确保在多次调用 evaluation_function 时不会重复添加处理器
if not logger.handlers:
    file_handler = logging.FileHandler('logs.log', mode='w', encoding='utf-8')
    logger.addHandler(file_handler)
    formatter = logging.Formatter('%(asctime)s | %(levelname)s | %(message)s')
    file_handler.setFormatter(formatter)
logger.setLevel(logging.INFO) # 设置日志级别

TIME_STEPS = 168

def get_known(key):
    """获取配置变量"""
    if key == 'datacenter_id':
        return ['DC1', 'DC2', 'DC3', 'DC4']
    elif key == 'actions':
        return ['buy', 'hold', 'move', 'dismiss']
    elif key == 'server_generation':
        return ['CPU.S1', 'CPU.S2', 'CPU.S3', 'CPU.S4', 'GPU.S1', 'GPU.S2', 'GPU.S3']
    elif key == 'latency_sensitivity':
        return ['high', 'medium', 'low']
    elif key == 'required_columns':
        return ['time_step', 'datacenter_id', 'server_generation', 'server_id', 'action']
    elif key == 'time_steps':
        return 168
    elif key == 'datacenter_fields':
        return ['datacenter_id', 'cost_of_energy', 'latency_sensitivity', 'slots_capacity']
    elif key == 'price_strategy_columns':
        return ['time_step', 'latency_sensitivity', 'server_generation', 'price']
    return None

def fleet_data_preparation(solution, servers, datacenters, selling_prices):
    """舰队数据准备"""
    # 检查数据格式
    check_data_format(solution)
    check_actions(solution)
    check_datacenters_servers_generation(solution)
    check_server_usage_by_release_time(solution, servers)

    # 删除重复的服务器ID
    solution = drop_duplicate_server_ids(solution)

    return solution

def check_data_format(solution):
    """检查数据格式是否包含所有必需的列"""
    required_columns = get_known('required_columns')
    missing_columns = [col for col in required_columns if col not in solution.columns]
    if missing_columns:
        raise ValueError(f"缺少必需的列: {missing_columns}")

    # 检查数据类型
    if not pd.api.types.is_integer_dtype(solution['time_step']):
        raise ValueError("time_step 必须是整数类型")

def check_actions(solution):
    """检查是否只使用了允许的动作"""
    allowed_actions = get_known('actions')
    invalid_actions = solution[~solution['action'].isin(allowed_actions)]['action'].unique()
    if len(invalid_actions) > 0:
        raise ValueError(f"使用了不允许的动作: {invalid_actions}")

def check_datacenters_servers_generation(solution):
    """检查数据中心和服务器世代是否按要求命名"""
    allowed_datacenters = get_known('datacenter_id')
    allowed_generations = get_known('server_generation')

    invalid_datacenters = solution[~solution['datacenter_id'].isin(allowed_datacenters)]['datacenter_id'].unique()
    invalid_generations = solution[~solution['server_generation'].isin(allowed_generations)]['server_generation'].unique()

    if len(invalid_datacenters) > 0:
        raise ValueError(f"使用了不允许的数据中心: {invalid_datacenters}")
    if len(invalid_generations) > 0:
        raise ValueError(f"使用了不允许的服务器世代: {invalid_generations}")

def check_server_usage_by_release_time(solution, servers):
    """检查在特定时间步只使用可购买的服务器"""
    # Create a dictionary for faster lookup of release_time for each server_generation
    servers['release_time_tuple'] = servers['release_time'].apply(eval)
    release_times_map = servers.set_index('server_generation')['release_time_tuple'].to_dict()

    # Filter solution for 'buy' actions
    buy_actions_df = solution[solution['action'] == 'buy'].copy()

    if not buy_actions_df.empty:
        # Map release times to the buy_actions_df
        buy_actions_df['release_range'] = buy_actions_df['server_generation'].map(release_times_map)

        # Identify invalid buys
        # A server is invalid if release_range is None (server_gen not found)
        # OR if time_step is outside the release_range [start, end]
        invalid_buys = buy_actions_df[
            (buy_actions_df['release_range'].isna()) |
            (buy_actions_df.apply(lambda row: not (row['release_range'][0] <= row['time_step'] <= row['release_range'][1]), axis=1))
        ]

        if not invalid_buys.empty:
            # Collect unique invalid server_gen and time_step combinations for error message
            error_details = invalid_buys[['server_generation', 'time_step']].drop_duplicates().to_string(index=False)
            raise ValueError(f"在不可购买的时间步使用了服务器:\n{error_details}")

def drop_duplicate_server_ids(solution):
    """删除使用相同服务器ID多次购买的服务器"""
    # Find buy actions with duplicate server_id
    buy_actions = solution[solution['action'] == 'buy']
    duplicate_servers_df = buy_actions[buy_actions.duplicated('server_id', keep='first')]

    if not duplicate_servers_df.empty:
        duplicate_server_ids = duplicate_servers_df['server_id'].unique()
        logger.warning(f"发现重复购买的服务器ID，将保留第一次买入记录，删除后续重复记录: {duplicate_server_ids}")
        
        # Get indices of first 'buy' for each server_id
        first_buy_indices = buy_actions.drop_duplicates(subset=['server_id'], keep='first').index
        
        # Get indices of all non-'buy' actions
        non_buy_indices = solution[solution['action'] != 'buy'].index
        
        # Combine indices and select rows
        solution_cleaned = solution.loc[first_buy_indices.union(non_buy_indices)].sort_values(by=['time_step', 'server_id']).reset_index(drop=True)
        return solution_cleaned
    
    return solution

def pricing_data_preparation(prices):
    """定价数据准备"""
    if prices is None or prices.empty:
        logger.warning("定价数据为空或None。")
        return pd.DataFrame(columns=get_known('price_strategy_columns'))

    # 检查定价策略必需列
    required_columns = get_known('price_strategy_columns')
    missing_columns = [col for col in required_columns if col not in prices.columns]
    if missing_columns:
        raise ValueError(f"定价策略缺少必需的列: {missing_columns}")

    return prices

def change_elasticity_format(elasticity):
    """调整弹性数据帧格式"""
    if elasticity is None or elasticity.empty:
        logger.warning("弹性数据为空或None。")
        return pd.DataFrame(columns=['server_generation', 'latency_sensitivity', 'elasticity'])

    # 确保elasticity有正确的列
    required_cols = ['server_generation', 'latency_sensitivity', 'elasticity']
    if not all(col in elasticity.columns for col in required_cols):
        raise ValueError("弹性数据缺少必需的列")

    return elasticity

def change_selling_prices_format(selling_prices):
    """调整售价数据帧格式"""
    if selling_prices is None or selling_prices.empty:
        logger.warning("售价数据为空或None。")
        return pd.DataFrame(columns=['server_generation', 'latency_sensitivity', 'selling_price'])

    # 确保selling_prices有正确的列
    required_cols = ['server_generation', 'latency_sensitivity', 'selling_price']
    if not all(col in selling_prices.columns for col in required_cols):
        raise ValueError("售价数据缺少必需的列")

    return selling_prices

def get_actual_demand(demand: pd.DataFrame) -> Dict[int, Dict[Tuple[str, str], float]]:
    """获取实际需求 - 保持不变"""
    actual_demand = {}
    for time_step in range(1, TIME_STEPS + 1):
        time_demand = {}
        step_data = demand[demand['time_step'] == time_step]

        if not step_data.empty:
            for _, row in step_data.iterrows():
                latency = row['latency_sensitivity']
                for sg in get_known('server_generation'):  # 使用已知的服务器世代
                    if sg in row and pd.notna(row[sg]):
                        key = (sg, latency)
                        time_demand[key] = float(row[sg]) # 确保为浮点数类型
        actual_demand[time_step] = time_demand
    return actual_demand

def get_time_step_prices(pricing_strategy, ts):
    """获取特定时间步的价格"""
    if pricing_strategy.empty:
        return pd.DataFrame(columns=get_known('price_strategy_columns'))

    return pricing_strategy[pricing_strategy['time_step'] == ts].copy() # 返回副本

def update_selling_prices(selling_prices_df, ts_prices_df):
    """根据定价策略更新售价"""
    if ts_prices_df.empty:
        return selling_prices_df.copy()

    # 创建一个临时DataFrame，包含需要更新的价格，并重命名列
    temp_prices_to_update = ts_prices_df[['server_generation', 'latency_sensitivity', 'price']].rename(columns={'price': 'selling_price'})

    # 使用 merge 进行更新， suffixes 用于区分合并前后的列
    updated_df = selling_prices_df.merge(
        temp_prices_to_update,
        on=['server_generation', 'latency_sensitivity'],
        how='left',
        suffixes=('_original', '_new')
    )
    # 合并新旧价格列：如果_new有值，则使用_new，否则使用_original
    updated_df['selling_price'] = updated_df['selling_price_new'].fillna(updated_df['selling_price_original'])
    
    # 删除临时列
    updated_df = updated_df.drop(columns=['selling_price_original', 'selling_price_new'])
    
    return updated_df

def update_demand_according_to_prices(ts_demand: Dict[Tuple[str, str], float], selling_prices_df: pd.DataFrame, base_prices_df: pd.DataFrame, elasticity_df: pd.DataFrame) -> Dict[Tuple[str, str], float]:
    """根据新价格更新需求"""
    if not ts_demand or selling_prices_df.empty or base_prices_df.empty or elasticity_df.empty:
        return ts_demand.copy()

    updated_demand = ts_demand.copy()

    # 创建一个合并的DataFrame，包含p0, p1, e，便于快速查找
    price_elasticity_data = base_prices_df.rename(columns={'selling_price': 'p0'}).merge(
        selling_prices_df.rename(columns={'selling_price': 'p1'}),
        on=['server_generation', 'latency_sensitivity'],
        how='left'
    ).merge(
        elasticity_df.rename(columns={'elasticity': 'e'}),
        on=['server_generation', 'latency_sensitivity'],
        how='left'
    )
    
    # 填充NaN值：如果新价格缺失，使用基准价格；如果弹性缺失，默认为0
    price_elasticity_data['p0'] = price_elasticity_data['p0'].fillna(0)
    price_elasticity_data['p1'] = price_elasticity_data['p1'].fillna(price_elasticity_data['p0']) 
    price_elasticity_data['e'] = price_elasticity_data['e'].fillna(0)

    # 转换为字典，以便通过 (server_gen, latency_sensitivity) 快速查找
    price_elasticity_map = {
        (row['server_generation'], row['latency_sensitivity']): {'p0': row['p0'], 'p1': row['p1'], 'e': row['e']}
        for _, row in price_elasticity_data.iterrows()
    }
    
    for (server_gen, latency_sensitivity), demand_value in ts_demand.items():
        price_info = price_elasticity_map.get((server_gen, latency_sensitivity))

        if price_info:
            p0 = price_info['p0']
            p1 = price_info['p1']
            e = price_info['e']

            if p0 != 0:
                price_change_percent = p1 / p0 - 1
                new_demand_val = demand_value * (1 + price_change_percent * e)
            else:
                new_demand_val = demand_value
            updated_demand[(server_gen, latency_sensitivity)] = max(0, math.ceil(new_demand_val))
        else:
            updated_demand[(server_gen, latency_sensitivity)] = demand_value
    #print(f"更新后的需求: \n{updated_demand}")
    return updated_demand

def get_maintenance_cost(base_fee, current_age, expected_life):
    """计算当前维护成本"""
    if expected_life <= 0 or current_age <= 0:
        return base_fee

    usage_ratio = current_age / expected_life

    # 维护成本 = 管理费 * [1 + 1.5 * 使用比率 * log_2(1.5 * 使用比率)]
    if usage_ratio > 0:
        log_term_arg = 1.5 * usage_ratio
        if log_term_arg > 1e-9: # 避免log(0)或负数，使用小epsilon
            log_term = 1.5 * usage_ratio * math.log2(log_term_arg)
            maintenance_cost = base_fee * (1 + log_term)
        else:
            maintenance_cost = base_fee
    else:
        maintenance_cost = base_fee

    return maintenance_cost

def check_datacenter_slots_size_constraint(fleet_df, datacenters, servers):
    """检查数据中心插槽大小约束"""
    if fleet_df.empty:
        return True

    server_slots_map = servers.set_index('server_generation')['slots_size'].to_dict()
    datacenter_capacity_map = datacenters.set_index('datacenter_id')['slots_capacity'].to_dict()

    if 'server_generation' not in fleet_df.columns:
        raise ValueError("fleet_df 缺少 'server_generation' 列，无法检查插槽约束。")
    
    fleet_df_copy = fleet_df.copy() # 操作副本
    fleet_df_copy['calculated_slots_size'] = fleet_df_copy['server_generation'].map(server_slots_map)

    # 按数据中心分组并求和插槽使用情况
    slot_usage_df = fleet_df_copy.groupby('datacenter_id')['calculated_slots_size'].sum().reset_index()
    slot_usage_df.columns = ['datacenter_id', 'used_slots']

    # 检查是否超过容量限制
    for _, row in slot_usage_df.iterrows():
        datacenter_id = row['datacenter_id']
        used_slots = row['used_slots']
        capacity = datacenter_capacity_map.get(datacenter_id)

        if capacity is not None and used_slots > capacity:
            raise ValueError(f"数据中心 {datacenter_id} 插槽使用量 {used_slots} 超过容量 {capacity}")

    return True

def adjust_capacity_by_failure_rate(base_capacity):
    """计算故障率f的辅助函数"""
    try:
        # 使用截断威布尔分布采样故障率 f ∈ [0.05, 0.1]
        # 参数设置需要根据实际需求调整
        c = 2.0  # 
        a = 0.05  # 下界
        b = 0.1   # 上界
        scale = b-a  # 尺度参数

        # 生成截断威布尔分布的故障率
        failure_rate = truncweibull_min.rvs(c, a, b, scale=scale, size=1)[0]
        
        # 确保在 [0.05, 0.1] 范围内
        failure_rate = np.clip(failure_rate, 0.05, 0.1)
        
        # 实际容量 = (1-f) * 容量
        actual_capacity = (1 - failure_rate) * base_capacity
        
        return math.floor(actual_capacity)
    except Exception as e:
        print("adjust_capacity_by_failure_rate 出错:", e)
        return 0 

def get_revenue(ts_demand: Dict[Tuple[str, str], float], capacity_df: pd.DataFrame, selling_prices_df: pd.DataFrame) -> float:
    """计算收入"""
    if not ts_demand or capacity_df.empty or selling_prices_df.empty:
        return 0.0

    # 将 ts_demand 转换为 DataFrame，便于向量化操作
    demand_items = []
    for (server_gen, latency_sensitivity), demand_value in ts_demand.items():
        demand_items.append({
            'server_generation': server_gen,
            'latency_sensitivity': latency_sensitivity,
            'demand_value': demand_value
        })
    demand_df = pd.DataFrame(demand_items)

    if demand_df.empty:
        return 0.0

    # 考虑故障率影响容量
    
    for my_index in  capacity_df.index:
        capacity_df.at[my_index, 'capacity'] = adjust_capacity_by_failure_rate(capacity_df.at[my_index, 'capacity'])

        
    
    # 将需求与容量合并
    merged_df = demand_df.merge(capacity_df, on=['server_generation', 'latency_sensitivity'], how='left')
    merged_df['capacity'] = merged_df['capacity'].fillna(0) # 如果没有对应容量，填充0

    # 计算实际服务的需求
    merged_df['served'] = np.minimum(merged_df['demand_value'], merged_df['capacity'])

    # 将合并后的数据与售价合并
    merged_df = merged_df.merge(selling_prices_df, on=['server_generation', 'latency_sensitivity'], how='left')
    merged_df['selling_price'] = merged_df['selling_price'].fillna(0) # 如果没有对应售价，填充0

    # 计算总收入
    total_revenue = (merged_df['served'] * merged_df['selling_price']).sum()

    return total_revenue



def get_evaluation(fleet,
                   pricing_strategy,
                   demand,
                   datacenters,
                   servers,
                   selling_prices,
                   elasticity,
                   time_steps=168,
                   verbose=1):
    """解决方案评估"""

    # 解决方案数据准备 (只处理一次，包含所有时间步的动作)
    fleet_plan = fleet_data_preparation(fleet.copy(), servers, datacenters, selling_prices)

    # 定价和需求数据准备
    pricing_strategy = pricing_data_preparation(pricing_strategy)
    elasticity = change_elasticity_format(elasticity)
    selling_prices = change_selling_prices_format(selling_prices)
    base_prices = selling_prices.copy() # 基准价格用于弹性计算

    # 需求数据准备 (预处理成字典，更快查找)
    demand_per_time_step = get_actual_demand(demand)
    #print("预处理后的需求数据:\n", demand_per_time_step)

    # 预处理静态数据，转换为字典或索引DataFrame以便快速查找
    servers_info_map = servers.set_index('server_generation')
    datacenters_info_map = datacenters.set_index('datacenter_id')

    # active_fleet_list: 存储当前时间步活跃的服务器信息
    active_fleet_list: List[Dict[str, Any]] = []

    total_objective = 0.0
    current_selling_prices=base_prices.copy() # 初始化当前售价为基准售价

    for ts in range(1, time_steps + 1):
        if verbose > 0:
            logger.info(f"处理时间步 {ts}")

        # 1. 获取当前时间步的需求
        ts_demand = demand_per_time_step.get(ts, {})
        #print(f"时间步 {ts} 的需求: \n{ts_demand}") 

        # 2. 更新售价和需求弹性
        ts_prices_df = get_time_step_prices(pricing_strategy, ts)
        current_selling_prices = update_selling_prices(current_selling_prices, ts_prices_df)
        ts_demand = update_demand_according_to_prices(ts_demand, current_selling_prices, base_prices, elasticity)
        #print(f"时间步 {ts} 的更新后需求: \n{ts_demand}")
        # 3. 处理当前时间步的舰队动作 (buy, move, dismiss)
        ts_actions = fleet_plan[fleet_plan['time_step'] == ts]
        #print(f"时间步 {ts} 的动作: \n{ts_actions}")
        current_ts_purchase_cost = 0.0
        current_ts_moving_cost = 0.0

        # 创建一个需要解雇的服务器ID集合，以便高效查找
        servers_to_dismiss_at_ts = set(ts_actions[ts_actions['action'] == 'dismiss']['server_id'])

        # 从 active_fleet_list 中过滤掉被解雇的服务器
        active_fleet_list = [s for s in active_fleet_list if s['server_id'] not in servers_to_dismiss_at_ts]

        # 处理购买和移动动作
        for _, action_row in ts_actions[ts_actions['action'] != 'dismiss'].iterrows():
            server_id = action_row['server_id']
            server_gen = action_row['server_generation']
            action = action_row['action']
            datacenter_id = action_row['datacenter_id']

            server_static_attrs = servers_info_map.loc[server_gen].to_dict() # 获取服务器的静态属性

            if action == 'buy':
                # 检查服务器是否已在活跃舰队中 (理论上由于 drop_duplicate_server_ids 不会发生)
                if not any(s['server_id'] == server_id for s in active_fleet_list):
                    active_fleet_list.append({
                        'server_id': server_id,
                        'server_generation': server_gen,
                        'datacenter_id': datacenter_id,
                        'buy_time': ts, # 记录购买时间
                        'slots_size': server_static_attrs['slots_size'],
                        'life_expectancy': server_static_attrs['life_expectancy'],
                        'energy_consumption': server_static_attrs['energy_consumption'],
                        'purchase_price': server_static_attrs['purchase_price'],
                        'cost_of_moving': server_static_attrs['cost_of_moving'],
                        'average_maintenance_fee': server_static_attrs['average_maintenance_fee'],
                        'capacity': server_static_attrs['capacity'] # 基础容量
                    })
                    current_ts_purchase_cost += server_static_attrs['purchase_price']
            elif action == 'move':
                # 在 active_fleet_list 中找到服务器并更新其 datacenter_id
                found_server = False
                for s in active_fleet_list:
                    if s['server_id'] == server_id:
                        s['datacenter_id'] = datacenter_id
                        current_ts_moving_cost += s['cost_of_moving']
                        found_server = True
                        break
                if not found_server:
                    logger.warning(f"服务器 {server_id} 在时间步 {ts} 尝试 'move' 但未在活跃舰队中找到。已忽略。")
        
        # 4. 将 active_fleet_list 转换为 DataFrame 以进行向量化计算
        current_fleet_df = pd.DataFrame(active_fleet_list) if active_fleet_list else pd.DataFrame(columns=[
            'server_id', 'server_generation', 'datacenter_id', 'buy_time', 'slots_size', 'life_expectancy',
            'energy_consumption', 'purchase_price', 'cost_of_moving', 'average_maintenance_fee', 'capacity'
        ])
        #print(f"当前时间步 {ts} 的舰队信息:\n", current_fleet_df)
        # 5. 检查数据中心插槽约束
        if not current_fleet_df.empty:
            try:
                check_datacenter_slots_size_constraint(current_fleet_df, datacenters, servers)
            except ValueError as e:
                logger.error(f"时间步 {ts} 违反插槽约束: {e}")
                raise e # 重新抛出异常以终止评估

        # 6. 计算当前时间步的容量 (考虑故障率)
        ts_capacity_df = pd.DataFrame(columns=['server_generation', 'latency_sensitivity', 'capacity'])
        if not current_fleet_df.empty:
            # 与数据中心信息合并，获取 latency_sensitivity
            temp_capacity_df = current_fleet_df.merge(
                datacenters_info_map[['latency_sensitivity']],
                on='datacenter_id',
                how='left'
            )
            #print(f"当前时间步 {ts} 的服务器信息:\n", temp_capacity_df)
            # 按服务器世代和延迟敏感度分组求和容量
            ts_capacity_df = temp_capacity_df.groupby(['server_generation', 'latency_sensitivity'])['capacity'].sum().reset_index()


        # 7. 计算收入
        current_ts_revenue = get_revenue(ts_demand, ts_capacity_df, current_selling_prices)
        total_objective += current_ts_revenue

        # 8. 计算能源和维护成本
        current_ts_energy_cost = 0.0
        current_ts_maintenance_cost = 0.0

        if not current_fleet_df.empty:
            # 能源成本
            current_ts_energy_cost = (current_fleet_df['energy_consumption'] * current_fleet_df['datacenter_id'].map(datacenters_info_map['cost_of_energy'])).sum()
            
            # 维护成本
            current_fleet_df['current_age'] = ts - current_fleet_df['buy_time'] + 1
            current_ts_maintenance_cost = current_fleet_df.apply(
                lambda row: get_maintenance_cost(row['average_maintenance_fee'], row['current_age'], row['life_expectancy']),
                axis=1
            ).sum()
        
        # 9. 从总目标中减去成本
        total_objective -= (current_ts_purchase_cost + current_ts_moving_cost + current_ts_energy_cost + current_ts_maintenance_cost)

        # 10. 更新寿命并移除达到预期寿命的服务器 (为下一个时间步做准备)
        servers_to_expire = set()
        for s in active_fleet_list:
            if (ts - s['buy_time'] + 1) >= s['life_expectancy']:
                servers_to_expire.add(s['server_id'])

        active_fleet_list = [s for s in active_fleet_list if s['server_id'] not in servers_to_expire]

        if verbose > 1:
            current_ts_profit = current_ts_revenue - (current_ts_purchase_cost + current_ts_moving_cost + current_ts_energy_cost + current_ts_maintenance_cost)
            logger.info(f"时间步 {ts} 利润: {current_ts_profit:.2f}")

    if verbose > 0:
        logger.info(f"总目标值: {total_objective:.2f}")

    return total_objective

def evaluation_function(fleet,
                        pricing_strategy,
                        demand,
                        datacenters,
                        servers,
                        selling_prices,
                        elasticity,
                        time_steps=168,
                        seed=None,
                        verbose=0):
    """
    评估Tech Arena Phase 1问题的解决方案。

    Parameters
    ----------
    fleet : pandas DataFrame
        服务器舰队。由参与者提供。
    pricing_strategy : pandas DataFrame
        定价策略。由参与者提供。
    demand : pandas DataFrame
        需求数据。默认在data文件夹中提供。
    datacenters : pandas DataFrame
        数据中心数据。默认在data文件夹中提供。
    servers : pandas DataFrame
        服务器数据。默认在data文件夹中提供。
    selling_prices : pandas DataFrame
        售价数据。默认在data文件夹中提供。
    elasticity : pandas DataFrame
        需求价格弹性数据。默认在data文件夹中提供。
    time_steps : int
        需要评估解决方案的时间步数。
    seed : int
        随机种子。
    verbose : int
        详细程度。

    Return
    ------
    返回一个浮点数，表示在所有时间步上评估的目标函数O的值。
    如果无法评估解决方案，函数返回None。
    """
    # 设置随机种子
    if seed is not None:
        np.random.seed(seed)

    # 评估解决方案
    try:
        return get_evaluation(fleet,
                            pricing_strategy,
                            demand,
                            datacenters,
                            servers,
                            selling_prices,
                            elasticity,
                            time_steps=time_steps,
                            verbose=verbose)
    # 捕获异常
    except Exception as e:
        logger.error(f"评估失败: {e}")
        if verbose > 0:
            print(f"评估失败: {e}")
        return None


